/**
 * \file: signature.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include <stdlib.h>
#include <alloca.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <openssl/pem.h>
#include <openssl/err.h>

#include "encryption/signature.h"
#include "encryption/sdc_access.h"
#include "util/logger.h"

//---------------------------------- private member functions --------------------------------------------------------
#ifdef ALD_DAEMON_CODE

static error_code_t signature_open_decrypted_filestream(FILE **decr_fs_ptr, char **decr_buffer_ptr, FILE *encr_fs);

#endif
//--------------------------------------------------------------------------------------------------------------------

//---------------------------------- private attributes --------------------------------------------------------------
static EVP_MD_CTX mdctx;
//--------------------------------------------------------------------------------------------------------------------

//---------------------------------------- reading in and destroying keys --------------------------------------------
error_code_t signature_init(void)
{
	ERR_load_crypto_strings();
	return RESULT_OK;
}

void signature_deinit(void)
{
	ERR_free_strings();
}
//--------------------------------------------------------------------------------------------------------------------

//---------------------------------------- reading in and destroying keys --------------------------------------------
error_code_t signature_read_private_key(const char *path, EVP_PKEY **privkey_ptr)
{
	error_code_t result=RESULT_OK;
	RSA *rsa_priv_key;
	EVP_PKEY *pkey=NULL;
	FILE *fd;
	fd=fopen(path,"r");
	if (fd==NULL)
	{
		logger_log_error("Key file not found: %s", path);
		return RESULT_FILE_NOT_FOUND;
	}

	rsa_priv_key=PEM_read_RSAPrivateKey(fd, NULL,0,NULL);
	if (rsa_priv_key==NULL)
	{
		logger_log_error("Unable to read key from \'%s\': %s",path,ERR_reason_error_string(ERR_get_error()));
		result=RESULT_INVALID_KEY;
	}

	if (result==RESULT_OK)
	{
		pkey=EVP_PKEY_new();
		if (EVP_PKEY_assign_RSA(pkey,rsa_priv_key)!=1)
		{
			logger_log_error("Unable to read key from \'%s\': %s",path,ERR_reason_error_string(ERR_get_error()));
			result=RESULT_INVALID_KEY;
		}
	}

	fclose(fd);

	if (result==RESULT_OK)
		*privkey_ptr=pkey;
	else
		signature_destroy_key(pkey);

	return result;
}

#ifdef ALD_DAEMON_CODE

error_code_t signature_read_public_key(const char *path, EVP_PKEY **pubkey_ptr, bool encrypted)
{
	error_code_t result=RESULT_OK;
	RSA *rsa_pub_key;
	EVP_PKEY *pkey=NULL;
	FILE *fs=NULL;
	char *key_buffer=NULL;

	fs=fopen(path,"r");
	if (fs==NULL)
	{
		logger_log_error("Key file not found: %s",path);
		return RESULT_FILE_NOT_FOUND;
	}

	if (encrypted)
	{
		FILE *decr_fs=NULL;
		result=signature_open_decrypted_filestream(&decr_fs, &key_buffer, fs);
		if (result==RESULT_FILE_NOT_FOUND)
			logger_log_error("Key file not found: %s",path);
		else if (result!=RESULT_OK)
			logger_log_error("Unable to decrypt key file: %s",path);

		fclose(fs);
		fs=decr_fs;
	}

	if (result != RESULT_OK)
		return result;

	rsa_pub_key=PEM_read_RSA_PUBKEY(fs, NULL,0,NULL);
	if (rsa_pub_key==NULL)
	{
		logger_log_error("Unable to read key from \'%s\': %s",path,ERR_reason_error_string(ERR_get_error()));
		result=RESULT_INVALID_KEY;
	}

	if (result==RESULT_OK)
	{
		pkey=EVP_PKEY_new();
		if (EVP_PKEY_assign_RSA(pkey,rsa_pub_key)!=1)
		{
			logger_log_error("Unable to read key from \'%s\': %s",path,ERR_reason_error_string(ERR_get_error()));
			result=RESULT_INVALID_KEY;
		}
	}

	fclose(fs);

	if (key_buffer!=NULL)
		sdc_access_free_buffer(key_buffer);

	if (result==RESULT_OK)
		*pubkey_ptr=pkey;
	else
		signature_destroy_key(pkey);

	return result;
}

static error_code_t signature_open_decrypted_filestream(FILE **decr_fs_ptr, char **decr_buffer_ptr, FILE *encr_fs)
{
	int fd;
	struct stat stat_result;
	char *encr_buf;
	error_code_t result;
	size_t decr_buf_size=0;

	fd=fileno(encr_fs);
	if (fstat(fd,&stat_result)==-1)
		return RESULT_FILE_NOT_FOUND;

	//sorry for (char*) this is just for lin(t)isfaction
	encr_buf=(char *)alloca(stat_result.st_size);
	if (encr_buf==NULL)
		return RESULT_NORESOURCE;

	if (read(fd, encr_buf,stat_result.st_size)!=stat_result.st_size)
		return RESULT_FILE_NOT_FOUND;

	result=sdc_access_decrypt_memory_block(decr_buffer_ptr, &decr_buf_size, encr_buf, stat_result.st_size);
	if (result!=RESULT_OK)
		return result;

	//open as read only memory stream (r) in binary mode (b)
	*decr_fs_ptr=fmemopen(*decr_buffer_ptr,decr_buf_size, "rb");
	if (*decr_fs_ptr==NULL)
		return RESULT_NORESOURCE;
	else
		return RESULT_OK;
}

#endif

void signature_destroy_key(EVP_PKEY *key)
{
	if (key!=NULL)
		EVP_PKEY_free(key);
}
//--------------------------------------------------------------------------------------------------------------------

//---------------------------- creating, reading, writing, and destroying signatures --------------------------------
error_code_t signature_create_for_file(const char *path, char *signature, EVP_PKEY *privkey)
{
	size_t sig_len=0;
	error_code_t result=RESULT_OK;
	int fd;
	char buf[2048];
	ssize_t data_read;

	EVP_MD_CTX_init(&mdctx);

	fd=open(path,O_RDONLY);
	if (fd==-1)
	{
		logger_log_error("File to sign not found: %s",path);
		result=RESULT_FILE_NOT_FOUND;
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestSignInit(&mdctx, NULL, SIG_ALGORITHM, NULL, privkey)!=1)
		{
			logger_log_error("Signing failed during initialization: %s", ERR_reason_error_string(ERR_get_error()));
			result=RESULT_SIGNING_FAILED;
		}
	}

	while ((data_read=read(fd,buf,2048))>0 && result==RESULT_OK)
	{
		if (EVP_DigestSignUpdate(&mdctx, buf, (size_t)data_read)!=1)
		{
			logger_log_error("Signing failed while feeding data into the algorithm: %s",
					ERR_reason_error_string(ERR_get_error()));
			result=RESULT_SIGNING_FAILED;
		}
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestSignFinal(&mdctx, NULL, &sig_len)!=1)
		{
			logger_log_error("Signing failed during finalization: %s",ERR_reason_error_string(ERR_get_error()));
			result=RESULT_SIGNING_FAILED;
		}
	}

	if (result==RESULT_OK && sig_len != RESPONSE_SIGNATURE_SIZE_USED)
	{
		logger_log_error("The provided private key has the wrong bit size. It needs a size of %d bit.",PRIV_KEY_LEN_BITS);
		logger_log_error("A signature size of %d was created, a size of %d is expected.", sig_len,RESPONSE_SIGNATURE_SIZE_USED);
		result=RESULT_INVALID_KEY;
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestSignFinal(&mdctx, (unsigned char *)signature, &sig_len)!=1)
		{
			logger_log_error("Signing failed during finalization: %s, Code: %d",
					ERR_reason_error_string(ERR_get_error()), ERR_get_error());
			result=RESULT_SIGNING_FAILED;
		}
	}

	if (fd!=-1)
		close(fd);

	return result;
}

error_code_t signature_create_for_data_block(void *data, size_t data_size,char *signature,
		EVP_PKEY *privkey)
{
	size_t sig_len=0;
	error_code_t result=RESULT_OK;

	EVP_MD_CTX_init(&mdctx);

	logger_log_debug("SIGNATURE - Creating signature of data block (len=%d).",data_size);

	if (EVP_DigestSignInit(&mdctx, NULL,SIG_ALGORITHM, NULL, privkey)!=1)
	{
		logger_log_error("Signing failed during initialization: %s", ERR_reason_error_string(ERR_get_error()));
		result=RESULT_SIGNING_FAILED;
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestSignUpdate(&mdctx, data, data_size)!=1)
		{
			logger_log_error("Signing failed while feeding data into the algorithm: %s",
					ERR_reason_error_string(ERR_get_error()));
			result=RESULT_SIGNING_FAILED;
		}
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestSignFinal(&mdctx, NULL, &sig_len)!=1)
		{
			logger_log_error("Signing failed during finalization: %s",ERR_reason_error_string(ERR_get_error()));
			result=RESULT_SIGNING_FAILED;
		}
	}

	if (result==RESULT_OK && sig_len != RESPONSE_SIGNATURE_SIZE_USED)
	{
		logger_log_error("The provided private key has the wrong bit size. It needs a size of %d bit.",PRIV_KEY_LEN_BITS);
		result=RESULT_INVALID_KEY;
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestSignFinal(&mdctx, (unsigned char *)signature, &sig_len)!=1)
		{
			logger_log_error("Signing failed during finalization: %s, Code: %d",
					ERR_reason_error_string(ERR_get_error()), ERR_get_error());
			result=RESULT_SIGNING_FAILED;
		}
	}

	return result;
}

error_code_t signature_read_from_file(const char *path, char *signature)
{
	error_code_t result=RESULT_OK;
	struct stat stat_result;
	size_t sig_size;
	int fd;

	//the file is one big blob representing the complete signature, we are taking the file size to determine the blob size
	if (stat(path,&stat_result)==-1)
	{
		logger_log_error("Signature file not found at \'%s\'- %s.", path,strerror(errno));
		return RESULT_FILE_NOT_FOUND;
	}
	sig_size=(size_t)stat_result.st_size;

	if (sig_size != RESPONSE_SIGNATURE_SIZE_USED)
	{
		logger_log_error("Signature file has invalid size. Expecting a signature size of %d bytes.",
				RESPONSE_SIGNATURE_SIZE_USED);
		return RESULT_SIGNATURE_FILE_INVALID;
	}

	//opening the file
	fd=open(path,O_RDONLY);
	if (fd==-1)
	{
		logger_log_error("Unable to load signature from \'%s\'.", path);
		return RESULT_FILE_NOT_FOUND;
	}

	if (read(fd,signature, sig_size)<(ssize_t)sig_size)
	{
		logger_log_error("Signature file corrupt: \'%s\'.", path);
		result=RESULT_SIGNATURE_FILE_INVALID;
	}

	close(fd);

	return result;
}

error_code_t signature_write_to_file(const char *path, char *signature)
{
	error_code_t result=RESULT_OK;
	int fd;

	(void)unlink(path);
	fd=open(path,O_CREAT|O_WRONLY,0666);
	if (fd==-1)
	{
		logger_log_error("Failed to create signature file \'%s\'.",path);
		return RESULT_SIGNATURE_FILE_INVALID;
	}

	if (write(fd,signature, RESPONSE_SIGNATURE_SIZE_USED)!=(ssize_t)RESPONSE_SIGNATURE_SIZE_USED)
	{
		logger_log_error("Failed to create signature file \'%s\'.",path);
		result=RESULT_SIGNATURE_FILE_INVALID;
	}

	close(fd);

	if (result!=RESULT_OK)
		(void)unlink(path);
	return result;
}
//--------------------------------------------------------------------------------------------------------------------

//---------------------------- checking files against signatures ----------------------------------------------------
error_code_t signature_check_file(const char *path, char *signature, EVP_PKEY *pubkey)
{
	error_code_t result=RESULT_OK;
	int fd;
	char buf[2048];
	ssize_t data_read;

	EVP_MD_CTX_init(&mdctx);

	logger_log_debug("SIGNATURE - Checking file \'%s\' against signature.",path);

	fd=open(path,O_RDONLY);
	if (fd==-1)
	{
		logger_log_debug("Unable to read in the file to check: %s",path);
		result=RESULT_FILE_NOT_FOUND;
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestVerifyInit(&mdctx, NULL, SIG_ALGORITHM, NULL, pubkey)!=1)
		{
			logger_log_debug("SIGNATURE - Failed to initialize the verification: %s",ERR_reason_error_string(ERR_get_error()));
			result=RESULT_VERIFICATION_FAILED;
		}
	}

	while ((data_read=read(fd,buf,2048))>0 && result==RESULT_OK)
	{
		if (EVP_DigestVerifyUpdate(&mdctx, buf, (size_t)data_read)!=1)
		{
			logger_log_debug("SIGNATURE - Failed to feed data into the verification algorithm: %s",
					ERR_reason_error_string(ERR_get_error()));
			result=RESULT_VERIFICATION_FAILED;
		}
	}

	if (result==RESULT_OK)
	{
		//sorry, bad API. Need to cast away const
		if (EVP_DigestVerifyFinal(&mdctx, (unsigned char *)signature,RESPONSE_SIGNATURE_SIZE_USED)!=1)
		{
			logger_log_debug("SIGNATURE - Failed to finalize the verification: %s",
					ERR_reason_error_string(ERR_get_error()));
			result=RESULT_VERIFICATION_FAILED;
		}
	}

	if (fd!=-1)
		close(fd);

	logger_log_debug("SIGNATURE - Checked file. Result code: %d",(int)result);

	return result;
}

error_code_t signature_check_data_block(const void *data, size_t data_size, const char *signature, EVP_PKEY *pubkey)
{
	error_code_t result=RESULT_OK;

	EVP_MD_CTX_init(&mdctx);

	logger_log_debug("SIGNATURE - Checking data(len=%d) block against signature.", data_size);

	if (EVP_DigestVerifyInit(&mdctx, NULL, SIG_ALGORITHM, NULL, pubkey)!=1)
	{
		logger_log_debug("SIGNATURE - Failed to initialize the verification: %s",ERR_reason_error_string(ERR_get_error()));
		result=RESULT_VERIFICATION_FAILED;
	}

	if (result==RESULT_OK)
	{
		if (EVP_DigestVerifyUpdate(&mdctx, data, data_size)!=1)
		{
			logger_log_debug("SIGNATURE - Failed to feed data into the verification algorithm: %s",
					ERR_reason_error_string(ERR_get_error()));
			result=RESULT_VERIFICATION_FAILED;
		}
	}

	if (result==RESULT_OK)
	{
		//sorry, bad API. Need to cast away const
		if (EVP_DigestVerifyFinal(&mdctx, (unsigned char *)signature,RESPONSE_SIGNATURE_SIZE_USED)!=1)
		{
			logger_log_debug("SIGNATURE - Failed to finalize the verification: %s",
					ERR_reason_error_string(ERR_get_error()));
			result=RESULT_VERIFICATION_FAILED;
		}
	}

	logger_log_debug("SIGNATURE - Checked data block. Result code: %d",(int)result);

	return result;
}
//--------------------------------------------------------------------------------------------------------------------
